/*
* Copyright (C) TES Electronic Solutions GmbH,
* All Rights Reserved.
* Contact: info@guiliani.de
*
* This file is part of the Guiliani HMI framework
* for the development of graphical user interfaces on embedded systems.
*/

#ifndef GUIANIMATION_ATTRIBUTE__H_
#define GUIANIMATION_ATTRIBUTE__H_

#include "GUIAnimation.h"
#include "GUIAnimatedAttribute.h"
#include "GUIEasing.h"

/**
   @brief This animation controls an attribute

    Example:
    @code

    eC_UInt uiDuration = 0; // The animation shall run infinitely.
    eC_Value vStartValue = eC_FromInt(500); // Start at 500
    eC_UInt uiOffTime = 300; // Off time is 300 ms.

    // Create the instance of the animation.
    CGUIAnimationAttribute* pkAnimation = new CGUIAnimationAttribute(pkAnimatedObject, uiDuration, CGUIAnimatedAttribute::ATTR_XPOS, vStartValue, vEndValue);

    // To start the animation call the StartAnimation() method of the animation.
    pAnimation->StartAnimation();

    @endcode
    @ingroup GUILIANI_ANIMATIONS

*/
class CGUIAnimationAttribute : public CGUIAnimation
{
public:
    /**
        Constructor
        @param pkAnimatedObject The Object that shall be animated.
        @param uiDuration The duration defines how long the animation will be running. Use 0 for endless animation.
        @param eTargetAttribute The attribute which will be modified
        @param eEasingType The easing-type for the animation
        @param vStartValue The start-value
        @param vTargetValue The target-value
    */
    CGUIAnimationAttribute(
        CGUIObject* pkAnimatedObject,
        const eC_UInt& uiDuration,
        const CGUIAnimatedAttribute::AnimatedAttribute_t& eTargetAttribute,
        const CGUIEasing::EasingType_t& eEasingType,
        const eC_Value& vStartValue,
        const eC_Value& vTargetValue);

#if defined(GUILIANI_STREAM_GUI)
    /**
        Default constructor for streaming.
    */
    CGUIAnimationAttribute();
#endif

   /**
    This function contains the animation logic. It typically accesses the animation's
    CGUIEasing functions to get the current values and sets these values in the animated object.
    This function is called by DoAnimate()
    @param uiElapsedTime Time elapsed (in ms) since start of animation.
    */
    void ApplyAnimation(eC_UInt uiElapsedTime);

    /**
        Makes sure that the object is visible when the animation is stopped or deleted.
        @param eStatus The new status of the animation.
    */
    void OnStatusChanged(AnimationStatus_t eStatus);

#if defined(GUILIANI_STREAM_GUI) || defined(GUILIANI_WRITE_GUI)
    /// The class version of the implementation.
    static const eC_UInt ANIMATION_ATTRIBUTE_CLASS_VERSION;
#endif

#ifdef GUILIANI_STREAM_GUI
    void ReadFromStream();
#endif

#ifdef GUILIANI_WRITE_GUI
    void WriteToStream(const eC_Bool bWriteClassID = false);
#endif
};

#endif
